using System;
using System.Data;
using System.Data.SqlClient;
using System.Reflection;
using gov.va.med.vbecs.ExceptionManagement;
using STOREDPROC = gov.va.med.vbecs.Common.VbecsStoredProcs;
using gov.va.med.vbecs.Common;

namespace gov.va.med.vbecs.DAL
{
	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Carl Jensen</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>11/1/2002</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>
	///	Data access layer class implementing certain actions 
	///	required to discard or quarantine blood unit.
	///</summary>

	public class DiscardOrQuarantineBloodUnit
	{

		#region Methods
		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/18/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="885"> 
		///		<ExpectedInput>Blood unit status data</ExpectedInput>
		///		<ExpectedOutput>None</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="1855"> 
		///		<ExpectedInput>Incorrect Row Version</ExpectedInput>
		///		<ExpectedOutput>RowVersionException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Updates the quarantine status of the blood unit in the database
		/// </summary>
		/// <param name="dtBloodUnitStatus">blood unit statuses</param>
		/// <param name="dtQuarantineExceptions">quarantine exceptions</param>
		/// <param name="dtWorkloadEvent"></param>
		/// <param name="dtOrderedUnit"></param>
		/// <param name="dtBCEMessages"></param>
		public static bool UpdateQuarantineStatus( DataTable dtBloodUnitStatus, DataTable dtWorkloadEvent, DataTable dtQuarantineExceptions, DataTable dtOrderedUnit, DataTable dtBCEMessages )
		{
			//CR 2941, CR 2945 Added dtBCEMessages
			DataTable[] dtArray =
			{
				dtBloodUnitStatus,
				dtQuarantineExceptions,
				dtWorkloadEvent,
				dtOrderedUnit,
				dtBCEMessages
			};

			string[] spArray = 
			{
				STOREDPROC.UpdateQuarantineStatus.StoredProcName,
				STOREDPROC.InsertExceptionQuarantine.StoredProcName,
				STOREDPROC.InsertWorkloadEvents.StoredProcName,
				STOREDPROC.ReleaseOrderedUnitsByBloodUnit.StoredProcName,
				STOREDPROC.HL7InsertMessageLog.StoredProcName
			};

			int returnCode = (new StoredProcedure()).TransactionalGetValue( spArray, dtArray );
			return (returnCode == 0);
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/18/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="886"> 
		///		<ExpectedInput>Blood unit identifier and status data</ExpectedInput>
		///		<ExpectedOutput>None</ExpectedOutput>
		///	</Case>
		///<Case type="1" testid ="1856"> 
		///		<ExpectedInput>Invalid parameters</ExpectedInput>
		///		<ExpectedOutput>DataAccessLayerException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Updates the discard status of the blood unit in the database
		/// </summary>
		/// <param name="dtBloodUnitFinance"></param>
		/// <param name="dtBloodUnitStatus"></param>
		/// <param name="dtDiscardExceptions"></param>
		/// <param name="dtWorkloadEvent"></param>
		/// <param name="dtBCEMessages"></param>
		public static bool UpdateDiscardStatus
		(
			DataTable dtBloodUnitFinance,
			DataTable dtBloodUnitStatus, 
			DataTable dtDiscardExceptions, 
			DataTable dtWorkloadEvent,
			DataTable dtBCEMessages
				)
		{
			//CR 2941, CR 2945 Added dtBCEMessages
			DataTable[] dtArray =
			{
				dtBloodUnitFinance,
				dtBloodUnitStatus,
				dtDiscardExceptions,
				dtWorkloadEvent,
				dtBCEMessages
			};

			string[] spArray = 
			{
				STOREDPROC.UpdateReturnCreditAmount.StoredProcName,
				STOREDPROC.UpdateDiscardStatus.StoredProcName,
				STOREDPROC.InsertExceptionQuarantine.StoredProcName,
				STOREDPROC.InsertWorkloadEvents.StoredProcName,
				STOREDPROC.HL7InsertMessageLog.StoredProcName
};

			//TODO: Code needs to be put in the sproc to handle rowversion, see UpdateQuarantineStatus
			int returnCode = (new StoredProcedure()).TransactionalGetValue(spArray, dtArray);
			return (returnCode == 0);
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/18/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1165"> 
		///		<ExpectedInput>Bloodunit Guid</ExpectedInput>
		///		<ExpectedOutput>DataRow</ExpectedOutput>
		///	</Case>
		///<Case type="1" testid ="1171"> 
		///		<ExpectedInput>Invalid BloodUnitGuid</ExpectedInput>
		///		<ExpectedOutput>RowNotInTableException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Returns unit information for DiscardQuarantine UC
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		/// <returns></returns>
		public static DataRow GetDiscardOrQuarantineData(System.Guid bloodUnitGuid)
		{
			SqlParameter[] prms = 
				{
					new SqlParameter(STOREDPROC.GetDiscardOrQuarantineData.bloodunitguid, SqlDbType.UniqueIdentifier),
					new SqlParameter(STOREDPROC.GetDiscardOrQuarantineData.divisioncode, SqlDbType.Char)
				};

			prms[0].Value = bloodUnitGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetDiscardOrQuarantineData.StoredProcName, prms);
			
			switch (ds.Tables[0].Rows.Count)
			{
				case 0:
					//No row was found for a blood unit from this division.
					throw new RowNotInTableException();
				case 1:
					return ds.Tables[0].Rows[0];
				default:
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.TooManyRecordsFoundException(MethodBase.GetCurrentMethod().Name).ResString;
					throw new DataAccessLayerException(errorMessage);
				}
			}
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/15/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1168"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable of status details</ExpectedOutput>
		///	</Case>
		///<Case type="1" testid ="1172"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Retrieves status details list needed to populate a combo box 
		/// </summary>
		/// <returns>DataTable containing status detail list</returns>
		public static DataTable GetStatusDetails()
		{
			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetStatusDetailsList.StoredProcName);

			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/28/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2472"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable of valid blood units</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2473"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a DataTable containing list of discarded/quarantined units 
		/// for a given division
		/// </summary>
		/// <returns>DataTable containing list of discarded/quarantined units</returns>
		public static DataTable GetDiscardOrQuarantineUnits()
		{
			SqlParameter[] prms = 
				{
					new SqlParameter(STOREDPROC.GetDiscardOrQuarantineUnits.divisioncode, Common.LogonUser.LogonUserDivisionCode)
				};
			
			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetDiscardOrQuarantineUnits.StoredProcName,prms);

			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>01/24/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6524"> 
		///		<ExpectedInput>Valid blood unit guid</ExpectedInput>
		///		<ExpectedOutput>DataSet</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6525"> 
		///		<ExpectedInput>Invalid blood unit guid</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Note: We must get at least one table back; the log-in/shipment ABO/Rh
		/// Will throw exception otherwise
		/// </summary>
		/// <returns></returns>
		public static DataSet GetAboRhForConsistencyCheck(Guid bloodUnitGuid)
		{
			if (bloodUnitGuid.Equals(Guid.Empty))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("bloodUnitGuid").ResString);
			}
			//
			SqlParameter[] prms = 
			{
				new SqlParameter(STOREDPROC.GetAboRhForConsistencyCheck.bloodunitguid, SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetAboRhForConsistencyCheck.divisioncode, SqlDbType.Char)
			};
			//
			prms[0].Value = bloodUnitGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;
			//
			DataSet dsReturn = Common.StoredProcedure.GetData(STOREDPROC.GetAboRhForConsistencyCheck.StoredProcName, prms);
			//
			// We must get at least one table back; the log-in/shipment ABO/Rh
			if(dsReturn == null || dsReturn.Tables.Count < 1 || dsReturn.Tables[0].Rows.Count == 0)
			{
				throw new DataAccessLayerException((Common.StrRes.SysErrMsg.Common.NoDataRowsFoundException(STOREDPROC.GetAboRhForConsistencyCheck.StoredProcName).ResString));
			}
			//
			return dsReturn;
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>01/24/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6526"> 
		///		<ExpectedInput>Valid blood unit guid</ExpectedInput>
		///		<ExpectedOutput>DataSet (may be empty)</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6527"> 
		///		<ExpectedInput>Invalid blood unit guid</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Note: may return no data if Antigen Typing has not been performed
		/// </summary>
		/// <returns></returns>
		public static DataSet GetAntigenTypingForConsistencyCheck(Guid bloodUnitGuid)
		{
			if (bloodUnitGuid.Equals(Guid.Empty))
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidFormat("bloodUnitGuid").ResString);
			}
			//
			SqlParameter[] prms = 
			{
				new SqlParameter(STOREDPROC.GetAntigenTypingForConsistencyCheck.bloodunitguid, SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetAntigenTypingForConsistencyCheck.divisioncode, SqlDbType.Char)
			};
			//
			prms[0].Value = bloodUnitGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;
			//
			return  Common.StoredProcedure.GetData(STOREDPROC.GetAntigenTypingForConsistencyCheck.StoredProcName, prms);
		}

		#endregion
	}
}